/* C Library example program
 *
 *      This example program REQUIRES an E1432 at logical address 8 in
 *      a VXI mainframe controlled by an embedded computer or MXI computer.
 *
 *      This sample program creates a group containing two channels
 *      and makes a block measurment with the group.

--------------------- example makefile ----------------------------------------
E1432_LIB=     /opt/e1432/lib
E1432_INC=     /opt/e1432/include

CC	= cc
LD	= cc
CFLAGS	= -Aa -D_HPUX_SOURCE -I $(E1432_INC) +O3 +w1
LDFLAGS	= -s

example:	$(E1432_LIB)/lib1432.a example.o
	$(LD) $(LDFLAGS) example.o $(E1432_LIB)/lib1432.a -lsicl -o example

example.o:	$(E1432_INC)/e1432.h

-------------------------------------------------------------------------------
*/

#include <stdio.h>
#include "e1432.h"

#define MODULE_ADDRESS	8	/* Logical Address for the E1432 module */
#define RANGE		10.0	/* 10 volt range */
#define BLOCKSIZE	16	/* 16 points per block */
#define NUM_CHANS	2	/* Two channels in group */
#define	MOD_COUNT	1	/* One module */

int
main(void)
{
    struct e1432_hwconfig hwconfig[1];
    SHORTSIZ16 laddr_list[1];
    SHORTSIZ16 channellist[NUM_CHANS];
    E1432ID hwid;
    SHORTSIZ16 groupon;
    SHORTSIZ16 error;
    LONGSIZ32 actualsize;
    FLOATSIZ32 buffer[NUM_CHANS][BLOCKSIZE];
    SHORTSIZ16 i, b;

    laddr_list[0] = MODULE_ADDRESS;
    channellist[0] = 1;
    channellist[1] = 4;

    /* Initialize the library */
    error = e1432_init_io_driver();
    if (error)
	(void) printf("e1432_init_io_driver() failed, error: %d\n", error);

    /* Change this 0 to 1 to see call tracing */
    e1432_trace_level(1);

    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */
    error = e1432_get_hwconfig(MOD_COUNT, laddr_list, hwconfig);
    if (error != 0)
    {
	error = e1432_install(MOD_COUNT, laddr_list, 0,
			      "/opt/e1432/lib/sema.bin");
	if (error)
	    (void) printf("e1432_install failed, error: %d\n", error);
    }

    /* Turn on debugging prints, a good idea while developing.  But do
       it after the above e1432_get_hwconfig, because we know that
       might error if the firmware is not yet installed in the
       module. */
    (void) e1432_print_errors(1);

    /* Assign channels to modules */
    error = e1432_assign_channel_numbers(MOD_COUNT, laddr_list, &hwid);
    if (error)
	(void) printf("e1432_assign_channel_numbers() failed, error: %d\n",
		      error);

    /* Get groupID of a two channel group.  Note that failure of this
       function is indicated by a POSITIVE return value. */
    groupon = e1432_create_channel_group(hwid, NUM_CHANS, channellist);
    if (groupon >= 0)
	(void) printf("e1432_create_channel_group() failed, error: %d\n",
		      groupon);

    /* Set up parameters for reading a block */
    error = e1432_set_analog_input(hwid, groupon,
				   E1432_INPUT_MODE_VOLT,
				   E1432_INPUT_SOURCE_BNC,
				   E1432_ANTI_ALIAS_ANALOG_ON,
				   E1432_COUPLING_DC,
				   RANGE);
    if (error)
	(void) printf("e1432_set_analog_input() failed, error: %d\n", error);

    /* No trigger needed for this example */
    error = e1432_set_trigger(hwid, groupon, E1432_CHANNEL_OFF,
			      0.0, 0.0, 0.0, E1432_TRIGGER_SLOPE_POS,
			      E1432_TRIGGER_MODE_LEVEL);
    if (error)
	(void) printf("e1432_set_trigger() failed, error: %d\n", error);

    /* Set up data transfer parameters */
    error = e1432_set_data_format(hwid, groupon, BLOCKSIZE,
				  E1432_DATA_SIZE_16,
				  E1432_BLOCK_MODE,
				  E1432_APPEND_STATUS_OFF);
    if (error)
	(void) printf("e1432_set_data_format() failed, error: %d\n", error);

    /* Set the data port to VME, not Local Bus */
    error = e1432_set_data_port(hwid, groupon, E1432_SEND_PORT_VME);
    if (error)
	(void) printf("e1432_set_data_port() failed, error: %d\n", error);

    /* Top span alias protected data */
    error = e1432_set_decimation_filter(hwid, groupon,
					E1432_ONEPASS,
					E1432_ANTI_ALIAS_DIGITAL_ON,
					0);
    if (error)
	(void) printf("e1432_set_decimation_filter() failed, error:%d\n",
		      error);

    /* Set the span of the measurement.  This was also done by
       e1432_set_decimation_filter above, but is done here to
       demonstrate e1432_set_span(). */
    error = e1432_set_span(hwid, groupon, 20000.0);
    if (error)
	(void) printf("e1432_set_span() failed, error:%d\n", error);

    /* This program will trigger and arm the measurement */
    error = e1432_set_auto_trigger(hwid, groupon, E1432_MANUAL_TRIGGER);
    if (error)
	(void) printf("e1432_auto_trigger() failed, error: %d\n", error);
    error = e1432_set_auto_arm(hwid, groupon, E1432_MANUAL_ARM);
    if (error)
	(void) printf("e1432_auto_arm() failed, error: %d\n", error);

    /* Start the measurement */
    error = e1432_init_measure(hwid, groupon);
    if (error)
	(void) printf("e1432_init_measure() failed, error: %d\n", error);

    /* Arm */
    error = e1432_arm_measure(hwid, groupon, 0);
    if (error)
	(void) printf("e1432_arm_measure() failed, error: %d\n", error);

    /* Trigger */
    error = e1432_trigger_measure(hwid, groupon, 0);
    if (error)
	(void) printf("e1432_trigger_measure() failed, error: %d\n", error);

    /* Wait for group block ready */
    while (!(error = e1432_block_available(hwid, groupon))); /* Note ; */
    if (error < 0)
	(void) printf("e1432_block_available() failed, error: %d\n", error);

    /* Block available flag true means the all channels have a block
       ready.  Read all blocks of data by using the group ID */
    error = e1432_read_float32_data(hwid, groupon, E1432_TIME_DATA,
				    buffer[0], /* start of 2 channel buffer */
				    NUM_CHANS * BLOCKSIZE, NULL,
				    &actualsize);
    if (error)
	(void) printf("e1432_read_float32_data() failed, error: %d\n", error);
    /* Report if a full block wasn't read */
    if (actualsize != BLOCKSIZE * NUM_CHANS)
	(void) printf("group read actualsize=%d  BLOCKSIZE=%d\n",
		      actualsize, BLOCKSIZE);

    /* Print the collected data */
    /* Print header */
    (void) printf("point   ");
    for (i = 0; i < NUM_CHANS; i++)
	(void) printf("chan %1d     ", channellist[i]);
    (void) printf("\n");

    /* Print data */
    for (b = 0; b < BLOCKSIZE; b++)
    {
	(void) printf("%3d -   ", b);
	for (i = 0; i < NUM_CHANS; i++)
	    (void) printf("%8.5f   ", buffer[i][b]);
	(void) printf("\n");
    }

    return 0;
}
